<?php
/**
 * Plugin Name: Centro de Descargas (completo)
 * Description: Centro de descargas: CPT "Descargas" + categorías, archivo, fecha, tamaño automático, icono (builtin o subido), shortcode por categoría, descarga forzada + contador de descargas.
 * Version: 1.1
 * Author: Tu Nombre
 * Text Domain: centro-descargas
 */

if (!defined('ABSPATH')) exit;

// Define plugin dir/url
define('CD_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('CD_PLUGIN_URL', plugin_dir_url(__FILE__));

/* -----------------------
   Register CPT & taxonomy
   ----------------------- */
add_action('init', 'cd_register_cpt_and_tax');
function cd_register_cpt_and_tax(){
    // CPT
    $labels = array(
        'name' => 'Descargas',
        'singular_name' => 'Descarga',
        'add_new_item' => 'Añadir nueva descarga',
        'edit_item' => 'Editar descarga',
        'new_item' => 'Nueva descarga',
        'view_item' => 'Ver descarga',
        'search_items' => 'Buscar descargas',
        'not_found' => 'No hay descargas',
    );
    register_post_type('download', array(
        'labels' => $labels,
        'public' => true,
        'has_archive' => true,
        'show_in_rest' => true,
        'supports' => array('title','editor','author'),
        'menu_icon' => 'dashicons-download',
        'rewrite' => array('slug' => 'descargas'),
    ));

    // Taxonomy
    register_taxonomy('download_category','download', array(
        'labels' => array(
            'name' => 'Categorías de descargas',
            'singular_name' => 'Categoría de descarga',
        ),
        'hierarchical' => true,
        'show_in_rest' => true,
        'rewrite' => array('slug' => 'descargas-categoria'),
    ));
}

/* -----------------------
   Metabox: archivo, fecha, tamaño (auto), icono
   ----------------------- */
add_action('add_meta_boxes', 'cd_add_metabox');
function cd_add_metabox(){
    add_meta_box('cd_file_meta','Archivo de descarga','cd_metabox_callback','download','normal','high');
}

function cd_metabox_callback($post){
    wp_nonce_field('cd_save_meta','cd_meta_nonce');

    $file_id = get_post_meta($post->ID, '_cd_file_id', true);
    $file_url = $file_id ? wp_get_attachment_url($file_id) : '';
    $date = get_post_meta($post->ID, '_cd_date', true);
    $size = get_post_meta($post->ID, '_cd_size', true);
    $downloads = get_post_meta($post->ID, '_cd_downloads', true);
    if ($downloads === '') $downloads = 0;

    // Icon meta
    $icon_type = get_post_meta($post->ID, '_cd_icon_type', true); // 'builtin' or 'custom'
    if (!$icon_type) $icon_type = 'builtin';
    $icon_key = get_post_meta($post->ID, '_cd_icon_key', true); // e.g. pdf, zip, default
    if (!$icon_key) $icon_key = 'default';
    $icon_id = get_post_meta($post->ID, '_cd_icon_id', true); // attachment id for custom icon
    $icon_url = $icon_id ? wp_get_attachment_url($icon_id) : '';

    // Builtin icons list (keys must match files in plugin/icons/)
    $builtin = array('pdf','zip','doc','xls','jpg','default');

    ?>
    <p>
        <a id="cd-upload-button" class="button">Seleccionar / Subir archivo</a>
        <a id="cd-remove-button" class="button" <?php if(!$file_id) echo 'style="display:none;"'; ?>>Quitar</a>
    </p>

    <p id="cd-file-row" <?php if(!$file_id) echo 'style="display:none;"';?>>
        <strong>Archivo:</strong> <span id="cd-file-name"><?php echo $file_url ? esc_html(basename(get_attached_file($file_id))) : ''; ?></span><br>
        <strong>URL:</strong> <a id="cd-file-url" href="<?php echo esc_url($file_url); ?>" target="_blank"><?php echo $file_url ? esc_html($file_url) : ''; ?></a>
    </p>
    <input type="hidden" id="cd_file_id" name="cd_file_id" value="<?php echo esc_attr($file_id); ?>" />

    <p>
        <label for="cd_date">Fecha (visible en frontend):</label><br>
        <input type="date" id="cd_date" name="cd_date" value="<?php echo esc_attr($date ? $date : date('Y-m-d')); ?>" />
    </p>

    <p>
        <strong>Tamaño (auto):</strong> <span id="cd_size"><?php echo $size ? esc_html($size) : '—'; ?></span>
    </p>

    <p>
        <strong>Descargas:</strong> <span id="cd_downloads"><?php echo intval($downloads); ?></span>
    </p>

    <hr>

    <h4>Icono del archivo</h4>
    <p>
        <label><input type="radio" name="cd_icon_type" value="builtin" <?php checked($icon_type,'builtin'); ?> /> Usar icono integrado</label>
        <label style="margin-left:15px;"><input type="radio" name="cd_icon_type" value="custom" <?php checked($icon_type,'custom'); ?> /> Usar icono personalizado (subir)</label>
    </p>

    <div id="cd-builtin-icons" <?php if($icon_type!='builtin') echo 'style="display:none;"'; ?>>
        <p>Selecciona un icono predefinido (se incluyen iconos para pdf, zip, doc, xls, jpg y uno por defecto):</p>
        <div style="display:flex;gap:10px;flex-wrap:wrap;">
            <?php foreach($builtin as $key): 
                $icon_src = esc_url( CD_PLUGIN_URL . "icons/{$key}.svg" );
                $checked = ($icon_key === $key) ? 'border:2px solid #0073aa;padding:4px;border-radius:6px;' : 'padding:4px;border:1px solid #ddd;border-radius:6px;';
            ?>
                <label style="text-align:center;cursor:pointer;<?php echo $checked; ?>">
                    <input type="radio" name="cd_icon_key" value="<?php echo esc_attr($key); ?>" <?php checked($icon_key,$key); ?> style="display:block;margin-bottom:6px;" />
                    <img src="<?php echo $icon_src; ?>" alt="<?php echo esc_attr($key); ?>" style="width:48px;height:48px;display:block;margin-bottom:4px;" /><br>
                    <?php echo esc_html(strtoupper($key)); ?>
                </label>
            <?php endforeach; ?>
        </div>
    </div>

    <div id="cd-custom-icon" <?php if($icon_type!='custom') echo 'style="display:none;"'; ?>>
        <p>
            <a id="cd-upload-icon-button" class="button">Subir / Seleccionar icono</a>
            <a id="cd-remove-icon-button" class="button" <?php if(!$icon_id) echo 'style="display:none;"'; ?>>Quitar icono</a>
        </p>
        <div id="cd-icon-row" <?php if(!$icon_id) echo 'style="display:none;"';?>>
            <strong>Icono seleccionado:</strong><br>
            <img id="cd-icon-preview" src="<?php echo esc_url($icon_url ? $icon_url : ''); ?>" style="width:64px;height:64px;display:block;margin-top:6px;" />
        </div>
        <input type="hidden" id="cd_icon_id" name="cd_icon_id" value="<?php echo esc_attr($icon_id); ?>" />
    </div>

    <script>
    (function($){
        // File uploader
        var file_frame;
        $('#cd-upload-button').on('click', function(e){
            e.preventDefault();
            if (file_frame) { file_frame.open(); return; }
            file_frame = wp.media.frames.file_frame = wp.media({
                title: 'Seleccionar archivo',
                button: { text: 'Seleccionar' },
                multiple: false
            });
            file_frame.on('select', function(){
                var attachment = file_frame.state().get('selection').first().toJSON();
                $('#cd_file_id').val(attachment.id);
                $('#cd-file-name').text(attachment.filename || attachment.title || '');
                $('#cd-file-url').attr('href', attachment.url).text(attachment.url);
                $('#cd-file-row').show();
                $('#cd-remove-button').show();
            });
            file_frame.open();
        });
        $('#cd-remove-button').on('click', function(e){
            e.preventDefault();
            $('#cd_file_id').val('');
            $('#cd-file-name').text('');
            $('#cd-file-url').attr('href','').text('');
            $('#cd-file-row').hide();
            $(this).hide();
        });

        // Icon type switch
        $('input[name="cd_icon_type"]').on('change', function(){
            var val = $(this).val();
            if (val === 'builtin') {
                $('#cd-builtin-icons').show();
                $('#cd-custom-icon').hide();
            } else {
                $('#cd-builtin-icons').hide();
                $('#cd-custom-icon').show();
            }
        });

        // Custom icon uploader
        var icon_frame;
        $('#cd-upload-icon-button').on('click', function(e){
            e.preventDefault();
            if (icon_frame) { icon_frame.open(); return; }
            icon_frame = wp.media.frames.icon_frame = wp.media({
                title: 'Seleccionar icono',
                button: { text: 'Seleccionar' },
                multiple: false
            });
            icon_frame.on('select', function(){
                var attachment = icon_frame.state().get('selection').first().toJSON();
                $('#cd_icon_id').val(attachment.id);
                $('#cd-icon-preview').attr('src', attachment.url);
                $('#cd-icon-row').show();
                $('#cd-remove-icon-button').show();
            });
            icon_frame.open();
        });
        $('#cd-remove-icon-button').on('click', function(e){
            e.preventDefault();
            $('#cd_icon_id').val('');
            $('#cd-icon-preview').attr('src','');
            $('#cd-icon-row').hide();
            $(this).hide();
        });

        // Select highlight visual when choosing builtin icon
        $('input[name="cd_icon_key"]').on('change', function(){
            $('input[name="cd_icon_key"]').each(function(){
                var lbl = $(this).closest('label');
                lbl.css('border','1px solid #ddd').css('padding','4px');
            });
            $(this).closest('label').css('border','2px solid #0073aa').css('padding','4px');
        });
    })(jQuery);
    </script>

    <?php
}

add_action('save_post', 'cd_save_meta');
function cd_save_meta($post_id){
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
    if (!isset($_POST['cd_meta_nonce']) || !wp_verify_nonce($_POST['cd_meta_nonce'],'cd_save_meta')) {
        return;
    }
    if (get_post_type($post_id) != 'download') return;

    // File id
    $file_id = isset($_POST['cd_file_id']) ? intval($_POST['cd_file_id']) : 0;
    if ($file_id) {
        update_post_meta($post_id, '_cd_file_id', $file_id);
        $file_path = get_attached_file($file_id);
        if ($file_path && file_exists($file_path)){
            $size_bytes = filesize($file_path);
            update_post_meta($post_id, '_cd_size', cd_format_bytes($size_bytes));
        } else {
            update_post_meta($post_id, '_cd_size', 'Desconocido');
        }
    } else {
        delete_post_meta($post_id, '_cd_file_id');
        delete_post_meta($post_id, '_cd_size');
    }

    // Date
    if (isset($_POST['cd_date']) && $_POST['cd_date'] != '') {
        $date = sanitize_text_field($_POST['cd_date']);
        update_post_meta($post_id, '_cd_date', $date);
    } else {
        delete_post_meta($post_id, '_cd_date');
    }

    // Icon fields
    $icon_type = isset($_POST['cd_icon_type']) && in_array($_POST['cd_icon_type'], array('builtin','custom')) ? $_POST['cd_icon_type'] : 'builtin';
    update_post_meta($post_id, '_cd_icon_type', $icon_type);

    if ($icon_type === 'builtin') {
        $icon_key = isset($_POST['cd_icon_key']) ? sanitize_text_field($_POST['cd_icon_key']) : 'default';
        update_post_meta($post_id, '_cd_icon_key', $icon_key);
        delete_post_meta($post_id, '_cd_icon_id');
    } else {
        $icon_id = isset($_POST['cd_icon_id']) ? intval($_POST['cd_icon_id']) : 0;
        if ($icon_id) update_post_meta($post_id, '_cd_icon_id', $icon_id);
        else delete_post_meta($post_id, '_cd_icon_id');
    }
}

/* Human readable bytes */
function cd_format_bytes($bytes, $decimals = 2) {
    if ($bytes <= 0) return '0 B';
    $units = array('B','KB','MB','GB','TB');
    $pow = floor(log($bytes, 1024));
    $pow = min($pow, count($units) - 1);
    $bytes /= pow(1024, $pow);
    return round($bytes, $decimals) . ' ' . $units[$pow];
}

/* -----------------------
   Shortcode
   ----------------------- */
add_shortcode('download_category', 'cd_shortcode_download_category');
function cd_shortcode_download_category($atts){
    $atts = shortcode_atts(array(
        'slug' => '',
        'posts_per_page' => 100,
        'show_date' => 'true',
        'show_size' => 'true',
        'show_name' => 'true',
        'show_downloads' => 'true',
        'download_text' => 'Descargar',
        'show_icon' => 'true',
    ), $atts, 'download_category');

    if (!$atts['slug']) return '<p>Especifica el slug de la categoría: <code>[download_category slug=\"mi-categoria\"]</code></p>';

    $term = get_term_by('slug', $atts['slug'], 'download_category');
    if (!$term) return '<p>Categoría no encontrada: ' . esc_html($atts['slug']) . '</p>';

    $query = new WP_Query(array(
        'post_type' => 'download',
        'posts_per_page' => intval($atts['posts_per_page']),
        'tax_query' => array(
            array(
                'taxonomy' => 'download_category',
                'field' => 'slug',
                'terms' => $atts['slug']
            )
        ),
        'orderby' => 'date',
        'order' => 'DESC',
    ));

    if (!$query->have_posts()) return '<p>No hay archivos en esta categoría.</p>';

    // enqueue frontend styles
    add_action('wp_footer', 'cd_print_frontend_styles');

    ob_start();
    ?>
    <table class="cd-download-table" style="width:100%;border-collapse:collapse;">
        <thead>
            <tr style="text-align:left;border-bottom:1px solid #ddd;">
                <?php if($atts['show_icon']=='true'): ?><th style="width:60px"></th><?php endif; ?>
                <?php if($atts['show_name']=='true'): ?><th>Nombre</th><?php endif; ?>
                <?php if($atts['show_size']=='true'): ?><th style="white-space:nowrap">Tamaño</th><?php endif; ?>
                <?php if($atts['show_date']=='true'): ?><th style="white-space:nowrap">Fecha</th><?php endif; ?>
                <?php if($atts['show_downloads']=='true'): ?><th style="white-space:nowrap">Descargas</th><?php endif; ?>
                <th></th>
            </tr>
        </thead>
        <tbody>
        <?php while($query->have_posts()): $query->the_post();
            $post_id = get_the_ID();
            $file_id = get_post_meta($post_id, '_cd_file_id', true);
            $size = get_post_meta($post_id, '_cd_size', true);
            $date = get_post_meta($post_id, '_cd_date', true);
            $title = get_the_title();
            $downloads = intval(get_post_meta($post_id, '_cd_downloads', true));
            $download_label = esc_html($atts['download_text']);
            $download_url = esc_url( home_url( sprintf('/download/%d', $post_id) ) );

            // icon logic
            $icon_html = '';
            if ($atts['show_icon'] === 'true') {
                $icon_type = get_post_meta($post_id, '_cd_icon_type', true);
                if (!$icon_type) $icon_type = 'builtin';
                if ($icon_type === 'custom') {
                    $icon_id = get_post_meta($post_id, '_cd_icon_id', true);
                    if ($icon_id) {
                        $icon_src = wp_get_attachment_url($icon_id);
                        $icon_html = '<img src="'.esc_url($icon_src).'" alt="" style="width:40px;height:40px;object-fit:contain;" />';
                    }
                } else {
                    $icon_key = get_post_meta($post_id, '_cd_icon_key', true);
                    if (!$icon_key) $icon_key = 'default';
                    $icon_src = CD_PLUGIN_URL . 'icons/' . $icon_key . '.svg';
                    $icon_html = '<img src="'.esc_url($icon_src).'" alt="" style="width:40px;height:40px;object-fit:contain;" />';
                }
            }
            ?>
            <tr style="border-bottom:1px solid #f0f0f0;">
                <?php if($atts['show_icon']=='true'): ?><td><?php echo $icon_html; ?></td><?php endif; ?>
                <?php if($atts['show_name']=='true'): ?><td><?php echo esc_html($title); ?></td><?php endif; ?>
                <?php if($atts['show_size']=='true'): ?><td style="white-space:nowrap"><?php echo $size ? esc_html($size) : '—'; ?></td><?php endif; ?>
                <?php if($atts['show_date']=='true'): ?><td style="white-space:nowrap"><?php echo $date ? esc_html(date_i18n(get_option('date_format'), strtotime($date))) : esc_html(get_the_date()); ?></td><?php endif; ?>
                <?php if($atts['show_downloads']=='true'): ?><td style="white-space:nowrap"><?php echo $downloads; ?></td><?php endif; ?>
                <td style="text-align:right;white-space:nowrap">
                    <?php if($file_id): ?>
                        <a href="<?php echo $download_url; ?>" class="button" target="_blank" rel="noopener noreferrer"><?php echo $download_label; ?></a>
                    <?php else: ?>
                        <span style="color:#999">Sin archivo</span>
                    <?php endif; ?>
                </td>
            </tr>
        <?php endwhile; wp_reset_postdata(); ?>
        </tbody>
    </table>
    <?php
    return ob_get_clean();
}

function cd_print_frontend_styles(){
    // Lightweight styling for the table (only printed once via footer)
    echo '<style>
    .cd-download-table td, .cd-download-table th { padding:8px 6px; vertical-align:middle; }
    .cd-download-table img { display:block; }
    </style>';
}

/* -----------------------
   Forzar descarga / contador
   ----------------------- */
add_action('init', 'cd_add_rewrite_rule');
function cd_add_rewrite_rule(){
    add_rewrite_rule('^download/([0-9]+)/?','index.php?cd_download=$matches[1]','top');
    add_rewrite_tag('%cd_download%','([0-9]+)');
}

add_filter('query_vars', 'cd_query_vars');
function cd_query_vars($vars){
    $vars[] = 'cd_download';
    return $vars;
}

add_action('template_redirect', 'cd_maybe_serve_download');
function cd_maybe_serve_download(){
    $download_id = get_query_var('cd_download');
    if (!$download_id) return;

    $post_id = intval($download_id);
    $post = get_post($post_id);
    if (!$post || $post->post_type !== 'download') {
        status_header(404);
        exit('Archivo no encontrado');
    }

    $file_id = get_post_meta($post_id, '_cd_file_id', true);
    if (!$file_id) {
        status_header(404);
        exit('Archivo no disponible');
    }

    $file_path = get_attached_file($file_id);
    if (!$file_path || !file_exists($file_path)) {
        status_header(404);
        exit('Fichero no encontrado en servidor.');
    }

    // Increase counter
    $count = intval(get_post_meta($post_id, '_cd_downloads', true));
    $count++;
    update_post_meta($post_id, '_cd_downloads', $count);

    // Serve file
    while (ob_get_level()) ob_end_clean();

    $file_mime = wp_check_filetype($file_path);
    $mime = $file_mime['type'] ? $file_mime['type'] : 'application/octet-stream';
    $basename = basename($file_path);
    header('Content-Description: File Transfer');
    header('Content-Type: ' . $mime);
    header('Content-Disposition: attachment; filename="' . rawurldecode($basename) . '"');
    header('Content-Transfer-Encoding: binary');
    header('Content-Length: ' . filesize($file_path));
    header('Cache-Control: must-revalidate, post-check=0, pre-check=0');
    header('Pragma: public');
    header('Expires: 0');

    set_time_limit(0);
    readfile($file_path);
    exit;
}

/* Activation / Deactivation */
register_activation_hook(__FILE__, 'cd_plugin_activate');
function cd_plugin_activate(){
    cd_register_cpt_and_tax();
    cd_add_rewrite_rule();
    flush_rewrite_rules();
}

register_deactivation_hook(__FILE__, 'cd_plugin_deactivate');
function cd_plugin_deactivate(){
    flush_rewrite_rules();
}

/* Admin columns */
add_filter('manage_download_posts_columns','cd_download_columns');
function cd_download_columns($cols){
    $cols_before = array_slice($cols, 0, 2, true);
    $cols_after = array_slice($cols, 2, null, true);
    $new = array_merge($cols_before, array('cd_size' => 'Tamaño', 'cd_date' => 'Fecha', 'cd_downloads' => 'Descargas'), $cols_after);
    return $new;
}

add_action('manage_download_posts_custom_column','cd_download_custom_column',10,2);
function cd_download_custom_column($column, $post_id){
    if ($column == 'cd_size') {
        $size = get_post_meta($post_id, '_cd_size', true);
        echo $size ? esc_html($size) : '—';
    } elseif ($column == 'cd_date') {
        $d = get_post_meta($post_id, '_cd_date', true);
        if ($d) echo esc_html(date_i18n(get_option('date_format'), strtotime($d)));
        else echo esc_html(get_the_date('', $post_id));
    } elseif ($column == 'cd_downloads') {
        $count = intval(get_post_meta($post_id, '_cd_downloads', true));
        echo $count;
    }
}

/* Enqueue admin scripts */
add_action('admin_enqueue_scripts', 'cd_admin_enqueue');
function cd_admin_enqueue($hook){
    global $post;
    if (($hook == 'post-new.php' || $hook == 'post.php') && isset($post) && $post->post_type == 'download') {
        wp_enqueue_media();
        wp_enqueue_script('jquery');
    }
}

/* Readme: show quick help on plugin list row */
add_filter('plugin_row_meta', 'cd_plugin_row_meta', 10, 2);
function cd_plugin_row_meta($links, $file){
    if (strpos($file, basename(__FILE__)) !== false){
        $links[] = '<a href="'.admin_url('edit.php?post_type=download').'">Ir a Descargas</a>';
    }
    return $links;
}
